
#ifndef F_CPU
	#define F_CPU 16000000UL
#endif
#define BOOTLOADER_ADDRESS (0x3c00U)

#include <stdint.h>
#include <stdlib.h>
#include <avr/io.h>
#include <avr/interrupt.h>
#include <util/delay.h>

#include "../protocol.h"
#include "config.h"
#include "softpwm.h"


// --- helper macros ---



/// If defined, this enables a fancy startup-sequence.
/**
Delete this line to save code size.
*/
//#define ENABLE_FANCY_STARTUP 1
//#define ENABLE_FANCY_LIGHTS1 1
//#define ENABLE_FANCY_LIGHTS2 1
#define ENABLE_TEST_LIGHTS1 1

// display-data
uint16_t	brightnesslevels[8][3];	// 8 oder SEGM_COUNT??

	
// sensor-data
volatile uint16_t receivebrightness[8];

// UART
uint8_t GetUARTByte(void)
{
	/* Wait for data to be received */
	while ( !(UCSRA & (1<<RXC)) )
	;

	/* Get and return received data from buffer */
	return UDR;
}

#define IsUARTByteAvailible() (UCSRA & (1<<RXC)) 

void SendUARTByte(uint8_t msg)
{
	while ( !( UCSRA & (1<<UDRE)) )
		;
	PINON(BUNTICH_CHIP_TXENABLE_PORT,BUNTICH_CHIP_TXENABLE_NUMBER);	// enable
	PINON(UCSRA,TXC);
	UDR=msg;
	while ( !( UCSRA & (1<<TXC)) )
		;
	PINOFF(BUNTICH_CHIP_TXENABLE_PORT,BUNTICH_CHIP_TXENABLE_NUMBER);	// disable
}

void InitUART(void)
{
	UCSRA = 0x20;	// normal speed, no interrupt
	UCSRB = 0x18;	// Rx/Tx, 8 bit
	UCSRC = 0x86;	// 8 bit, no parity
	UBRRH = 0;	// 1Mbps
	UBRRL = 3;
}



void InitSensor(void)
{
	// CTC
	PINON(TCCR2,WGM21);
	PINOFF(TCCR2,WGM20);
	// no pwm
	PINOFF(TCCR2,COM21);
	PINOFF(TCCR2,COM20);
	// no prescaler
	PINON(TCCR2,CS20);
	PINOFF(TCCR2,CS21);
	PINOFF(TCCR2,CS22);
	// Compare-Register
	OCR2=64;
	// enable interrupt
//	PINON(TIMSK,OCIE2);
}

ISR(TIMER2_COMP_vect)
{
	uint16_t *ptr=receivebrightness;
	for(uint8_t i=0;i<8;i++)
	{
		if( (PINA & (1<<i)) && (*ptr<0xFFFF) )
		{
			(*ptr)++;
		}
		ptr++;
	}

}

// Init all ports at once, hard-coded
void InitPorts()
{
	// LED Ports
	DDRA = 0xff; // 1=output
	DDRB = 0xff;
	DDRC = 0xff;
	PORTA = 0xff; // 1 while input will activate pull-up, 1 = LED OFF
	PORTB = 0xff;
	PORTC = 0xff;
	
	// Other ports:
	// 7 IP TERM (Jumper)
	// 6 O1 ENUM2D
	// 5 O1 ENUMOUT
	// 4 IP ENUMIN
	// 3 O0 COMMONCONTROL (0 = LEDs activated)
	// 2 O0 TXENABLE (1 = enabled)
	// 1 O1 TX
	// 0 IP RX
	DDRD  = 0b01101110;
	PORTD = 0b11110011;
}

void brightnessupdate(void)
{
	for(uint8_t i=0;i<8;i++)
	{
		SoftPWMSetR(i,brightnesslevels[i][0]>>8);
		SoftPWMSetG(i,brightnesslevels[i][1]>>8);
		SoftPWMSetB(i,brightnesslevels[i][2]>>8);		
	}
	SoftPWMSwapBuffers();
}

int main(void)
{
	InitPorts();
	InitUART();
	SoftPWMInit();	
	InitSensor();
		
	
	// node adress
	uint8_t nodeAdress=BUNTICH_BROADCAST_ADDRESS;
	

	uint8_t *brightnesspointer; //=&(brightnesslevels[0][0]);
	
	sei();

#ifdef ENABLE_FANCY_LIGHTS1
{
	uint8_t lednr, colornr;
	int8_t updowns[8][3];
	
	for(lednr = 0; lednr<8; lednr++)
	{
		for(colornr=0; colornr<3; colornr++)
		{
			updowns[lednr][colornr]=(((long)random()) & 0x0003ffffL) * (random()&0x01000000L?1L:-1L) ;
		}
	}	
		
	while(!IsUARTByteAvailible())
	{
		for(lednr = 0; lednr<8; lednr++)
		{
			for(colornr=0; colornr<3; colornr++)
			{
				int32_t temp_l = (int32_t)brightnesslevels[lednr][colornr] + (int32_t)updowns[lednr][colornr];
				if((temp_l > 0x7fffL) || (temp_l < 0)) updowns[lednr][colornr] *= -1;
				brightnesslevels[lednr][colornr] += updowns[lednr][colornr];
				brightnesslevels[lednr][colornr] &= 0x7fff;
			}
		}		
		//_delay_ms(2);
		brightnessupdate();		
	}
}
#endif

#ifdef ENABLE_FANCY_LIGHTS2
{
	uint8_t lednr, colornr;
	int8_t updowns[8][3];
	uint16_t palette[8][3];
	
	for(lednr = 0; lednr<8; lednr++)
	{
			palette[lednr][0] = 0x7f00;
			palette[lednr][1] = (lednr) << 12;
			palette[lednr][2] = (lednr) << 12;
	}	
		
	while(!IsUARTByteAvailible())
	{
		for(lednr = 0; lednr<8; lednr++)
		{
				brightnesslevels[lednr][0] = palette[lednr][0];
				brightnesslevels[lednr][1] = palette[lednr][1];
				brightnesslevels[lednr][2] = palette[lednr][2];
		}		
		//_delay_ms(2);
		brightnessupdate();		
	}
}
#endif

#ifdef ENABLE_TEST_LIGHTS1
{
	uint8_t lednr, colornr = 0, blinkcycles = 4;
	int8_t updowns[8][3];
	uint16_t palette[8][3];
	int16_t counter1 = 50; 
	
	for(lednr = 0; lednr<8; lednr++)
	{
			palette[lednr][0] = 0x7f00;
			palette[lednr][1] = (lednr) << 12;
			palette[lednr][2] = (lednr) << 12;
	}	
	palette[0][0] = 0; palette[0][1] = 0; palette[0][2] = 0; 
	palette[1][0] = 0xff00; palette[1][1] = 0; palette[1][2] = 0; 
	palette[2][0] = 0; palette[2][1] = 0xff00; palette[2][2] = 0; 
	palette[3][0] = 0; palette[3][1] = 0; palette[3][2] = 0xff00; 
	palette[4][0] = 0; palette[4][1] = 0xff00; palette[4][2] = 0xff00; 
	palette[5][0] = 0xff00; palette[5][1] = 0xff00; palette[5][2] = 0; 
	palette[6][0] = 0xff00; palette[6][1] = 0; palette[6][2] = 0xff00; 
	palette[7][0] = 0xff00; palette[7][1] = 0xff00; palette[7][2] = 0xff00; 
		
	while(!IsUARTByteAvailible() && (blinkcycles>0))
	{
		for(lednr = 0; lednr<8; lednr++)
		{
				brightnesslevels[lednr][0] = palette[colornr][0];
				brightnesslevels[lednr][1] = palette[colornr][1];
				brightnesslevels[lednr][2] = palette[colornr][2];
		}		
		_delay_ms(2);
		brightnessupdate();		
		if(counter1-- == 0) {
			counter1 = 50;
			colornr++;
			if(colornr > 7) { 
                colornr = 0;
				blinkcycles--;
			}			
		}
	}
}
#endif
//while(1)
//{
	
	// fancy startup
#ifdef ENABLE_FANCY_STARTUP
#define FANCY_STARTUP_DELAY 5
for(;;)
{	
	// red
	for(uint8_t br=0;br<=SoftPWM_SLOT_COUNT;br++)
	{
		SoftPWMClearBuf();
		SoftPWMSetR(0,br);
		SoftPWMSwapBuffers();
		_delay_ms(FANCY_STARTUP_DELAY);
	}

	for(uint8_t i=0;i<8-1;i++)
	{
		uint8_t first=SoftPWM_SLOT_COUNT,second=0;
		while(first>0)
		{
			first--;
			second++;
			
			SoftPWMClearBuf();
			SoftPWMSetR(i,first);
			SoftPWMSetR(i+1,second);
			SoftPWMSwapBuffers();
			_delay_ms(FANCY_STARTUP_DELAY);
		}
	}
	
	// green
	for(uint8_t br=0;br<SoftPWM_SLOT_COUNT;br++)
	{
		SoftPWMClearBuf();
		SoftPWMSetG(0,br);
		SoftPWMSwapBuffers();
		_delay_ms(FANCY_STARTUP_DELAY);
		SoftPWMSetR(7,SoftPWM_SLOT_COUNT-br);
	}
	for(uint8_t i=0;i<8-1;i++)
	{
		uint8_t first=SoftPWM_SLOT_COUNT,second=0;
		while(first>0)
		{
			first--;
			second++;
			
			SoftPWMClearBuf();
			SoftPWMSetG(i,first);
			SoftPWMSetG(i+1,second);
			SoftPWMSwapBuffers();
			_delay_ms(FANCY_STARTUP_DELAY);
		}
	}
	
	// blue
	for(uint8_t br=0;br<SoftPWM_SLOT_COUNT;br++)
	{
		SoftPWMClearBuf();
		SoftPWMSetB(0,br);
		SoftPWMSwapBuffers();
		_delay_ms(FANCY_STARTUP_DELAY);
		SoftPWMSetG(7,SoftPWM_SLOT_COUNT-br);
	}
	for(uint8_t i=0;i<8-1;i++)
	{
		uint8_t first=SoftPWM_SLOT_COUNT,second=0;
		while(first>0)
		{
			first--;
			second++;
			
			SoftPWMClearBuf();
			SoftPWMSetB(i,first);
			SoftPWMSetB(i+1,second);
			SoftPWMSwapBuffers();
			_delay_ms(FANCY_STARTUP_DELAY);
		}
	}
}  
#endif
//}
	SoftPWMClearBuf();
	SoftPWMSetG(0,SoftPWM_SLOT_COUNT/2);
	SoftPWMSwapBuffers();
	
	/* Warte auf Adressbyte */
	while(1)
	{
		// Clear UART Error flags

		uint8_t uartbyte;
		while(    (( (uartbyte=GetUARTByte())) != nodeAdress) && (uartbyte != BUNTICH_BROADCAST_ADDRESS)  )
			;
		switch(GetUARTByte())
		{	
			/* Komplette Zeile empfangen und puffern */
			case BUNTICH_CMD_DISPLAY_DATA:
				// Get and store all led bytes
				brightnesspointer=&(brightnesslevels[0][0]);
				uint8_t i;
				for(i=0;i<24;i++)
					(*brightnesspointer++)=GetUARTByte();
				SendUARTByte(BUNTICH_RESPONSE_OK);
				
				// Wait for special segment number
//				while(segmentnumber!=MAGIC_SEGMENT_NUMBER)
//					;
				
				// Calculate lookup tables
				brightnesspointer=&(brightnesslevels[0][0]);
				for(i=0;i<8;i++)
				{
					SoftPWMSetR(i,*brightnesspointer++);
					SoftPWMSetG(i,*brightnesspointer++);
					SoftPWMSetB(i,*brightnesspointer++);
				}
				break;

			case BUNTICH_CMD_DISPLAY_SYNC:
				SoftPWMSwapBuffers();
				// no answer (broadcast)
				break;

			case BUNTICH_CMD_ENUMERATE_RESET:
				nodeAdress=BUNTICH_BROADCAST_ADDRESS;
				PINON(BUNTICH_CHIP_ENUMOUT_PORT,BUNTICH_CHIP_ENUMOUT_NUMBER);
				// no answer (broadcast)
				break;

			case BUNTICH_CMD_ENUMERATE_ENUM:
				if( ((~BUNTICH_CHIP_ENUMIN_PIN) & (1<<BUNTICH_CHIP_ENUMIN_NUMBER))   &&  (nodeAdress==BUNTICH_BROADCAST_ADDRESS) )
				{
/*					SoftPWMClearBuf();
					SoftPWMSetB(0,SoftPWM_SLOT_COUNT/2);
					SoftPWMSetG(1,SoftPWM_SLOT_COUNT/2);
					SoftPWMSwapBuffers();*/
					nodeAdress=GetUARTByte() | 0x80;
					PINOFF(BUNTICH_CHIP_ENUMOUT_PORT,BUNTICH_CHIP_ENUMOUT_NUMBER);
					if( BUNTICH_CHIP_TERM_PIN & 1<<BUNTICH_CHIP_TERM_NUMBER )	// not terminated
						SendUARTByte(BUNTICH_RESPONSE_OK);
					else
						SendUARTByte(BUNTICH_RESPONSE_TERMINATE);
					SoftPWMClearBuf();
					SoftPWMSetB(0,SoftPWM_SLOT_COUNT/2);
					SoftPWMSetB(1,SoftPWM_SLOT_COUNT/2);
					SoftPWMSwapBuffers();
				}
				break;

			case BUNTICH_CMD_SENSOR_START:
				SoftPWMDisable();
				PINON(COMMONCONTROL_PORT,COMMONCONTROL_NUMBER);	// set common-level to LOW (driver stage is inverting)
				DDR_RED=0xFF;
				PORT_RED=0xFF;
				for(uint8_t i=0;i<8;i++)
					receivebrightness[i]=1;
				DDR_RED=0x00;
				PORT_RED=0x00;
				PINON(TIMSK,OCIE2);
				break;

			case BUNTICH_CMD_SENSOR_STOP:
				PINOFF(TIMSK,OCIE2);
				PINOFF(COMMONCONTROL_PORT,COMMONCONTROL_NUMBER);	// set common-level to HIGH (driver stage is inverting)
				DDR_RED=0xFF;
				SoftPWMEnable();
				break;

			case BUNTICH_CMD_SENSOR_READ:
				for(uint8_t i=0;i<8;i++)
				{
					SendUARTByte( (receivebrightness[i]) >> 8);
					SendUARTByte( (receivebrightness[i]) & 0xFF);
				}
				break;
                
            case BUNTICH_CMD_BOOTLOADER:
                if(GetUARTByte() == BUNTICH_CMD_BOOTLOADER)
                {
                    // Do reset
                    cli();
                    void (*bl_restart)( void ) = BOOTLOADER_ADDRESS;
					bl_restart();
                }
                break;
		}
	}
	return 0;
}

