/**
\file softpwm.c
\brief This file contains all functions needed to use SoftPWM.
*/

#include <stdint.h>

#include <avr/io.h>
#include <avr/interrupt.h>
#include <avr/pgmspace.h>
#include "softpwm.h"

// SlotStates contains the value that is written to port at this time slot
// Double-buffering is used to avoid flickering.
// Slots are ordered not linear but in the order defined by SoftPWM_SlotOrder.
// Slot order is optimized (=linear) for interrupt routine.
rgb_t		    SoftPWM_SlotStates1[SoftPWM_SLOT_COUNT];	
rgb_t		    SoftPWM_SlotStates2[SoftPWM_SLOT_COUNT];

// This is the ordering of the slots. This is used when the slots
// are reprogrammed for setting a new color. 
const uint8_t   SoftPWM_SlotOrder[SoftPWM_SLOT_COUNT] PROGMEM = 
{ 
    SoftPWM_SLOT_ORDER 
};

// This contains the lengths of the slots
const uint16_t  SoftPWM_SlotLen[SoftPWM_SLOT_COUNT] PROGMEM =
{
	SoftPWM_SLOT_LEN
};

// Used by interrupt to cycle through SlotStates
uint8_t      *SoftPWM_SlotStatesOutPtr;
const prog_uint16_t     *SoftPWM_SlotLenPtr;

// Used by main to build up SlotStates table
uint8_t      *SoftPWM_SlotStatesInPtr;

// Flags for ping-pong buffer
uint8_t     SoftPWM_Flags;

/// SoftPWMInit initializes SoftPWM 
/**
This function enables Timer1, sets a compare value for CTC-mode, enables the correct output-ports and does some internal initialization.
*/
void SoftPWMInit(void)
{
    uint8_t tmp_sreg;  // store status register 
    tmp_sreg = SREG;   
    cli();             // Interrupts disable
	
	// init ports
	DDR_RED = DDR_BLUE = DDR_GREEN = 0xff;
    PORT_RED = PORT_BLUE = PORT_GREEN = 0xff;
	PINON(COMMONCONTROL_DDR, COMMONCONTROL_NUMBER);
	PINOFF(COMMONCONTROL_PORT, COMMONCONTROL_NUMBER);
	
	// init variables, buffers and pointers
    SoftPWM_Flags = 0;
	SoftPWM_SlotStatesOutPtr = &SoftPWM_SlotStates2[0][0];
	SoftPWM_SlotLenPtr       = SoftPWM_SlotLen;
	SoftPWM_SlotStatesInPtr  = &SoftPWM_SlotStates2[0][0];
	SoftPWMClearBuf();
	
	SoftPWM_SlotStatesInPtr  = &SoftPWM_SlotStates1[0][0];
	SoftPWMClearBuf();
	
    // WGM13..10 = 1100 (12)
    // TCCR1A = 0b01000000; // Toggle OC1A for debug
    TCCR1A = 0b00000000; // no output compare ports active
    TCCR1B = 0b00011010; // CLKIO / 8
    // TCCR1B = 0b00011011; // CLKIO / 64
    ICR1 = SoftPWM_TIMER_TOP_VALUE;    
    TIMSK &= 0b11000011;
    TIMSK |= 0b00001000;    // OCIE1B int enable
	
	OCR1B = 1;
	TCNT1 = 1;
    TIFR |= _BV(OCF1B);

	SREG = tmp_sreg;    // restore interrupt state
}


#if 0
/// This is the interrupt-vector for the timer0-compare-match-interrupt
ISR(TIMER1_COMPB_vect)
{
	uint16_t NextSegLen;
	
	PORT_RED=*(SoftPWM_SlotStatesOutPtr++);
	PORT_GREEN=*(SoftPWM_SlotStatesOutPtr++);
	PORT_BLUE=*(SoftPWM_SlotStatesOutPtr++);
	
	NextSegLen = pgm_read_word(SoftPWM_SlotLenPtr++);
	
	// if(NextSegLen < SoftPWM_NOINTLEAVELEN)
	
	NextSegLen += OCR1B;
	
	// If end reached, overflow and restart, change SlotBuffer here 
	if(NextSegLen > SoftPWM_TIMER_TOP_VALUE)
	{
		NextSegLen -= (SoftPWM_TIMER_TOP_VALUE+1);
	    SoftPWM_SlotLenPtr = SoftPWM_SlotLen;
	    if(SoftPWM_Flags & (1<<SoftPWM_FLAG_BUF1OUT))
	    {
		    SoftPWM_SlotStatesOutPtr = &SoftPWM_SlotStates1[0][0];
	    }
	    else
	    {
		    SoftPWM_SlotStatesOutPtr = &SoftPWM_SlotStates2[0][0];
	    }
	}
	
	OCR1B = NextSegLen;
}
#endif

/// Swaps the buffers for double-buffering
/**
*/
void SoftPWMSwapBuffers(void)
{
	if(SoftPWM_Flags & (1<<SoftPWM_FLAG_BUF1OUT))
	{
		PINOFF(SoftPWM_Flags, SoftPWM_FLAG_BUF1OUT);
		// use now buf2 for out, buf1 for in
		SoftPWM_SlotStatesInPtr = &SoftPWM_SlotStates1[0][0];
	}
	else
	{
		PINON(SoftPWM_Flags, SoftPWM_FLAG_BUF1OUT);		
		// use now buf1 for out, buf2 for in
		SoftPWM_SlotStatesInPtr = &SoftPWM_SlotStates2[0][0];
	}
}

/// sets the red level of a certain LED
/**
\param number	the number of the LED 0<=number<8
\param value	the new brightness-value for the specified LED; 0<=value<=SoftPWM_MAX_LEVEL
\warning number should never exceed the value 7, or it may lead to corrupted memory!

If value is bigger than SoftPWM_SLOT_COUNT-1, the brightness will be set to SoftPWM_SLOT_COUNT-1
*/
void SoftPWMSetR(uint8_t number,uint8_t value)
{
	value = value / ((SoftPWM_MAX_LEVEL+1)/(SoftPWM_SLOT_COUNT+1));
	number ^= 4;    // Patch for swapped LED stripes
	
	for(uint8_t i=0;i<SoftPWM_SLOT_COUNT;i++)
	{
		if(pgm_read_byte(&SoftPWM_SlotOrder[i])<value)
		 	PINOFF(*(SoftPWM_SlotStatesInPtr+3*i),number);
		//	PINOFF(SoftPWM_SlotStatesInPtr[i][0],number);
		else
		 	PINON(*(SoftPWM_SlotStatesInPtr+3*i),number);
		//	PINON(SoftPWM_SlotStatesInPtr[i][0],number);
	}
}

/// sets the green level of a certain LED
/**
\see SoftPWMSetR
*/
void SoftPWMSetG(uint8_t number,uint8_t value)
{
	value = value / ((SoftPWM_MAX_LEVEL+1)/(SoftPWM_SLOT_COUNT+1));
	number ^= 4;    // Patch for swapped LED stripes
	
	for(uint8_t i=0;i<SoftPWM_SLOT_COUNT;i++)
	{
		if(pgm_read_byte(&SoftPWM_SlotOrder[i])<value)
			PINOFF(*(SoftPWM_SlotStatesInPtr+3*i+1),number);
//			PINOFF(SoftPWM_SlotStatesInPtr[i][1],number);
		else
			PINON(*(SoftPWM_SlotStatesInPtr+3*i+1),number);
//			PINON(SoftPWM_SlotStatesInPtr[i][1],number);
	}
}

/// sets the blue level of a certain LED
/**
\see SoftPWMSetR
*/
void SoftPWMSetB(uint8_t number,uint8_t value)
{
	value = value / ((SoftPWM_MAX_LEVEL+1)/(SoftPWM_SLOT_COUNT+1));
	number ^= 4;    // Patch for swapped LED stripes
	
	for(uint8_t i=0;i<SoftPWM_SLOT_COUNT;i++)
	{
		if(pgm_read_byte(&SoftPWM_SlotOrder[i])<value)
			PINOFF(*(SoftPWM_SlotStatesInPtr+3*i+2),number);
//			PINOFF(SoftPWM_SlotStatesInPtr[i][2],number);
		else
			PINON(*(SoftPWM_SlotStatesInPtr+3*i+2),number);
//			PINON(SoftPWM_SlotStatesInPtr[i][2],number);
	}
}

/// Enable SoftPWM-output
void SoftPWMEnable(void)
{
	// enable Compare-interrupt
	PINON(TIMSK,OCIE1B);
}

/// Disable output
/**
Disables output and timer-interrupts, but will continue counting
*/
void SoftPWMDisable(void)
{
	// disable Compare-interrupt
	PINOFF(TIMSK,OCIE1B);
}

/// Clear the buffer that is currently writable
void SoftPWMClearBuf(void)
{
	uint8_t *ptr=SoftPWM_SlotStatesInPtr;
	for(uint16_t i=0;i<SoftPWM_SLOT_COUNT*3;i++,ptr++)
		*ptr=0xFF;
}
