
/**
 * Flaschenlampe ("Bottle-Lamp")
 * 
 * Copyright (c) 2018 Christian Tonhaeuser
 * This code is published under the MIT license. See LICENSE.txt for details.
 * 
 * This sketch will control a HighPower LED that is supplied directly from a
 * single cell LiPo battery via a BC337 NPN transistor.
 * 
 * The idea is to put the circuit into a small, sturdy case together with the LiPo
 * battery and the LED mounted in the top of the case. Then you can put a bottle
 * full of water on top of the case and have an improvised lamp for lighting up
 * your camping table at night so you can find your beer in the dark... ;-)
 * 
 * The sketch will measure the maximum current that is flowing through the led at
 * any time and adjust the PWM duty cycle accordingly so that a constant current
 * flow (and therefore brightness) is guaranteed when battery voltage drops.
 * 
 * Additionally, the battery voltage is constantly monitored and the circuit will
 * be powered down if voltage goes below a defined threshold.
 * This would prevent unprotected LiPo batteries from deep discharge.
 * Nevertheless, I do NOT recommend using unprotected LiPo batteries for powering the lamp!
 * 
 * A single button will control the whole circuit.
 * Long presses turn the lamp on and off, short presses will toggle between low
 * and maximum brightness. Maximum brightness will gradually dim down as battery
 * voltage drops over time, low brightness will usually be pretty much constant until
 * the battery is almost completely drained.
 * 
 * This sketch has been tested with an ATTiny85, but it should also work with Tiny25/45.
 * See the .png file in the archive for information on the circuit.
 * 
 */



#define LED PB4                         // Pin controlling the LED PWM
#define CURRENT_SENSE A3                // Current sense pin
#define R_SENSE ((uint32_t)220)         // Current sense resistor value (milliohms)
#define SENSE_INTERVAL 1000             // Sense current every n PWM cycles.
                                        // Value of 1000 paired with 245Hz PWM freq means roughly every 4 seconds.

#define BUTTON PB0                      // Pin connected to the pushbutton
#define BUTTON_INT PCINT0               // Pin change interrupt for the pushbutton

#define VCC_OUT PB1                     // Pin driving the voltage divider
#define VCC_SENSE A1                    // Pin used for sensing Vcc voltage from divider
#define VCC_MIN ((uint32_t)3300)        // Shutdown voltage threshold in mV.
                                        // I recommend shut down under 3.3V, this will make things easier for the CCC (cheap chinese charger)
                                        // Also, if lamp is shut down due to undervoltage, 3.3V remaining will give an 
                                        // ample time buffer to recharge the battery.

#define VCC_SENSE_DIVIDER_LARGE 10000   // Value of bigger resistor in voltage divider (between VCC_OUT and VCC_SENSE pins)
#define VCC_SENSE_DIVIDER_SMALL 3300    // Value of smaller resistor in voltage divider (between VCC_SENSE and GND pins)


/* --- NO MORE CONFIG BELOW THIS POINT! --- */

#define NO_PRESS 0                  // Value returned if no key is pressed
#define SHORT_PRESS 1               // Value returned for a short keypress
#define LONG_PRESS 2                // Value returned for long keypress
#define LONG_PRESS_DURATION_MS 500  // Duration for a long keypress in ms


#include <avr/sleep.h>              // Sleep Modes
#include <avr/power.h>              // Power management



uint8_t pwmVal = 20;                // Stores pwm values, 20 is inital value after power is connected.

boolean maxCurrent = false;         // marker: Are we currently in max current mode or not?
uint16_t desiredCurrent = 75;       // mean current that we want to have (in mA).

volatile uint16_t waitCounter = 0;              // Counter for timing current sensing. Also used to force new ADC conversion.
volatile uint16_t sensedCurrent = 0;            // Contains the last read ADC value for current sensing.
volatile uint8_t currentReadingAvailable = 0;   // Flag for main loop to know that there's a new reading available.


/**
 * Check if button is currently pressed.
 * Return true if pressed, false otherwise.
 */
boolean isButtonPressed(){
  return (PINB & _BV(BUTTON)) == 0;     // Button is low-active

}

/**
 * Check for a keypress.
 * Keypresses can be either LONG or SHORT.
 * Note that for LONG keypresses, this function will return even if button is not released yet.
 * 
 * Return 0 if no key is pressed.
 * Return 1 for short (<500ms) keypress
 * Return 2 if key pressed >500ms
 */
uint8_t getKeypress(){
  if(!isButtonPressed()){
    return NO_PRESS; // Button not pressed, we can return immediately.
  }
  unsigned long ts = millis();  // Save a timestamp
  while(isButtonPressed()){
    if(millis() - ts >= LONG_PRESS_DURATION_MS){  // If button was pressed for more than LONG_PRESS_DURATION_MS...
      return LONG_PRESS;                          // ...return LONG_PRESS
    }
    delay(7);   // Just a small delay, could also leave that out...
  }
  // Button was depressed and released befor a LONG_PRESS was detected.
  if(millis()-ts > 50){   // We ignore the keypress if it was shorter than 50ms (aka. primitive debouncing)
    return SHORT_PRESS;   // Keypress was >50ms and <LONG_PRESS_DURATION_MS  --> return a SHORT_PRESS
  } else {
    return NO_PRESS;      // Keypress was <50ms, we ignore it.
  }
}


/**
 * Timer0 is used by Arduino libs --> better not touch
 * 
 * OK, we want the timer to run reasonably slow.
 * This will make it easier to get a correct current measurement from the PWM.
 * Running the timer with around 244Hz overflow frequency will give us ample time for ADC
 * 
 */
void initTimer(){
  GTCCR = _BV(PWM1B) | _BV(COM1B1);             // PWM mode, clear OC1B on compare match
  TIMSK |= _BV(TOIE1);                          // Enable overflow interrupt for Timer1
  OCR1C = 255;                                  // Set TOP to 255, this is the value the counter will count to before resetting.
  OCR1B = pwmVal;                               // Set last known value, will be updated soon, anyway.
  TCNT1 = 0;                                    // (Reset Timer)
  TCCR1 = _BV(CS12) | _BV(CS11) | _BV(CS10);    // Run timer with CLK/64 --> 244Hz PWM Freq.
}

/**
 * Timer1 Overflow Interrupt Routine
 * Main thing it does is to synchronize the current sensing to the PWM cycle.
 * This way, we can measure the momentary current the LED will draw during the on-phase
 * of the PWM then calculate the correct PWM duty cycle to reach the desired mean current.
 */
ISR(TIMER1_OVF_vect){
  waitCounter++;
  if(waitCounter > SENSE_INTERVAL){
    sensedCurrent = analogRead(CURRENT_SENSE);  // Get reading
    currentReadingAvailable = 1;                // Set flag that new reading is there
    waitCounter = 0;                            // reset counter
  }
}

/**
 * Helper method to force a new current reading.
 * Will be called when the brightness is changed via a short keypress.
 */
void forceCurrentSense(){
  waitCounter = SENSE_INTERVAL + 1;
}

/**
 * Sets the parameter as the new compare value for the timer.
 * Note: The timer on Tiny85 is glitch-free when in PWM mode, this means that
 * the value will only be set during the next timer overflow and thus not
 * cause any glitches in the PWM.
 * See Tiny85 datasheet for details, they explain it pretty good there...
 */
void setPWMValue(uint8_t value){
  OCR1B = value;
}

/**
 * Turn off the LED by setting PWM duty cycle to zero.
 */
void ledOff(){
  setPWMValue(0);
}

/**
 * Enable the ADC by setting the analogReference to the internal 1.1V reference.
 */
void initADC(){
  analogReference(INTERNAL);
  delay(5);  // give the internal ref some time to start up.
}

/**
 * We don't actually do anything during the Pin change interrupt, it is only
 * used to wake the MCU from power down mode.
 * However, without defining the interrupt vector, strange things would happen...
 * That's why there's an empty ISR here.
 */
ISR (PCINT0_vect) 
{
  // Nothing to do here...
}




/**
 * Check if voltage is below threshold.
 * Return true if so.
 */
boolean isVoltageTooLow(){
  PORTB |= _BV(VCC_OUT);    // Turn on voltage divider
  uint16_t reading = analogRead(VCC_SENSE);
  delay(1);
  reading = analogRead(VCC_SENSE);
  delay(1);
  reading = analogRead(VCC_SENSE);
  boolean result = (reading < (VCC_MIN*(uint32_t)VCC_SENSE_DIVIDER_SMALL/(uint32_t)(VCC_SENSE_DIVIDER_SMALL+VCC_SENSE_DIVIDER_LARGE))*(uint32_t)1023/(uint32_t)1100);
  PORTB &= ~_BV(VCC_OUT);
  return result;
}


/**
 * This method will handle sending the CPU to power down mode
 * as well as waking back up in an orderly fashion.
 */
void powerDown(){
  do {
    set_sleep_mode(SLEEP_MODE_PWR_DOWN);
    ACSR |= _BV(ACD);      // Disable Analog comparator
    power_all_disable ();  // power off ADC, Timer 0 and 1, serial interface
    sleep_enable();        // Enable sleep mode and...
    sleep_cpu();           // ...good night!
    // MCU is now asleep, execution will continue here on pin change interrupt!
    sleep_disable();       // disable sleep mode
    power_all_enable();    // restart peripherals
    
    initADC();             // Need ADC for voltage measurement
  } while(isVoltageTooLow() || getKeypress() != LONG_PRESS);  // Keep going to sleep if voltage is too low
                                                              // or if keypress was too short.
  
  initTimer();               // Voltage is fine and we had a long keypress --> Turn on LED again.
  while(isButtonPressed());  // Wait until button is released...
  forceCurrentSense();       // And check if our PWM duty cycle is still fine.

}


/**
 * Send CPU into idle mode.
 * CPU will wake up on any interrupt.
 */
void sleepIdle(){
  set_sleep_mode(SLEEP_MODE_IDLE);
  sleep_enable();
  sleep_cpu();
  sleep_disable();      
}

void setup() {
  pinMode(VCC_OUT, OUTPUT);
  pinMode(LED, OUTPUT);
  pinMode(CURRENT_SENSE, INPUT);
  pinMode(VCC_SENSE, INPUT);
  pinMode(BUTTON, INPUT_PULLUP);

  // pin change interrupt (example for D4)
  PCMSK  |= _BV(BUTTON_INT);  // want pin PB0 / pin 5
  GIFR   |= _BV(PCIF);    // clear any outstanding interrupts
  GIMSK  |= _BV(PCIE);    // enable pin change interrupts 

  powerDown();
}




void loop() {

  sleepIdle(); // Go into idle mode. 
               // CPU will wake up every timer overflow interrupt, i.e. fairly often.


  // Every time we get a new current measurement we adjust PWM value
  if(currentReadingAvailable == 1){   // New reading available?
    currentReadingAvailable = 0;      // Reset flag

    uint32_t milliVolts = (uint32_t)1100 * (uint32_t)sensedCurrent / (uint32_t)1023; // calculate the measured voltage...
    uint32_t current = milliVolts * (uint32_t)1000 / R_SENSE;                        // ...and transform it into a current.
    
    // Check if we can even reach the desired current anymore.
    // Keep in mind, the sensed current is also the maximum the LED can do with the remaining battery voltage!
    if(desiredCurrent > current){   // Is our goal unreachable?
      pwmVal = 255;                 // Yes, so let's go all out to get as close as possible
    } else {
      pwmVal = ((uint32_t)desiredCurrent * (uint32_t)255) / current ;  // Calculate PWM duty cycle
      if(pwmVal < 20){    // Lower duty cycles than ~20 can mess with current sensing...
        pwmVal = 20;      // ...so this is the lowest we'll go.
      }
    }
    setPWMValue(pwmVal);    // Immediately set the new value

    // While we're at it, let's check the voltage:
    if(isVoltageTooLow()){  // If the voltage is too low...
      powerDown();          // ...we go into powerdown mode.
    }
    
  }

  switch(getKeypress()){          // Read a keypress
    case LONG_PRESS:              // Long press:
      ledOff();                   // Immediately turn off the LED
      while(isButtonPressed());   // Wait until the button was released and...
      powerDown();                // ...power down the MCU
      setPWMValue(pwmVal);        // Once we wake up again we set the last PWM value again
      break;
      
    case SHORT_PRESS:             // Short press
      if(maxCurrent){             // We now toggle between High and low brightness
        desiredCurrent = 75;    // Minimum current is 75mA, which is bright enough to see by at night.
        maxCurrent = false;
      } else {
        desiredCurrent = 300;   // Maximum current is 300mA which is low enough to make sure the NPN is not damaged.
        maxCurrent = true;
      }
      forceCurrentSense();        // Since we just changed the desiredCurrent, force a current sense cyle
                                  // so that the PWM duty cycle is updated accordingly.
      break;
    default:
      break;
  }

  

}
