/**
\file
\brief This file contains all code for compilation under LINUX. If you are developing for windows only, this file can be deleted.
\warning Do not compile this file directly if you design a cross-platform application. Compile rs232lib.cpp instead.
\author hephaisto

This file is licensed under the Creative Commons License (Attribution-NonCommercial) Version 3.0
http://creativecommons.org/licenses/by-nc/3.0/
*/
#include <stdio.h>
#include <unistd.h>
#include <fcntl.h>
#include <errno.h>
#include <termios.h>

#include "rs232lib.h"

namespace rs232lib
{
using std::string;

Rs232::Rs232()
{
	opened=false;
}

void Rs232::Open(const string path, const int baudrate)
{
	if(opened)
		throw Rs232Exception("this instance of Rs232 has already an opened port");
	fd = open(path.c_str(), O_RDWR | O_NOCTTY | O_NDELAY);
	if(fd==-1)
		throw Rs232Exception("unable to open port");
	// disable blocking behaviour (sets read to return immediately if buffer is empty)
	if(fcntl(fd,F_SETFL, FNDELAY)==-1)
		throw Rs232Exception("unable to set file flags");
//	fcntl(fd,F_SETFL, 0);
	
	termios options;
	if(tcgetattr(fd, &options)==-1)
		throw Rs232Exception("unable to read attributes");
//	memset(&options, 0, sizeof(options));
		
	// set baudrate
	int baud_internal;
	switch(baudrate)
	{
		case      50 :
			baud_internal = B50;
			break;
		case      75 :
			baud_internal = B75;
			break;
		case     110 :
			baud_internal = B110;
			break;
		case     134 :
			baud_internal = B134;
			break;
		case     150 :
			baud_internal = B150;
			break;
		case     200 :
			baud_internal = B200;
			break;
		case     300 :
			baud_internal = B300;
			break;
		case     600 :
			baud_internal = B600;
			break;
		case    1200 :
			baud_internal = B1200;
			break;
		case    1800 :
			baud_internal = B1800;
			break;
		case    2400 :
			baud_internal = B2400;
			break;
		case    4800 :
			baud_internal = B4800;
			break;
		case    9600 :
			baud_internal = B9600;
			break;
		case   19200 :
			baud_internal = B19200;
			break;
		case   38400 :
			baud_internal = B38400;
			break;
		case   57600 :
			baud_internal = B57600;
			break;
		case  115200 :
			baud_internal = B115200;
			break;
		case  230400 :
			baud_internal = B230400;
			break;
		case  460800 :
			baud_internal = B460800;
			break;
		case  500000 :
			baud_internal = B500000;
			break;
		case  576000 :
			baud_internal = B576000;
			break;
		case  921600 :
			baud_internal = B921600;
			break;
		case 1000000 :
			baud_internal = B1000000;
			break;
		default:
			throw Rs232Exception("unsupported baudrate");
	}
	
	// set receiver on, local mode
	options.c_cflag |= ( CLOCAL | CREAD );
	
	// set parity off, 8 bit
	options.c_cflag &= ~PARENB;
	options.c_cflag &= ~CSTOPB;
	options.c_cflag &= ~CSIZE;
	options.c_cflag |= ~CS8;
	
	// disable hardware flow control (needs additional CTS/RTS lines)
	options.c_cflag &= ~CRTSCTS;
	
	// raw input mode, no echo
	options.c_lflag &= ~(ICANON | ECHO | ECHOE | ISIG );
	
	// disable software flow control
	options.c_iflag &= ~( IXON | IXOFF | IXANY );
	// disable parity check
	options.c_iflag &= ~( INPCK );
	
	// raw output mode
	options.c_oflag &= ~OPOST;
	
	if(cfsetispeed(&options, baud_internal)==-1)
		throw Rs232Exception("unable to set baudrate (input)");
	if(cfsetospeed(&options, baud_internal)==-1)
		throw Rs232Exception("unable to set baudrate (output)");
	
	// apply configuration
	if(tcsetattr(fd, TCSANOW, &options)==-1)
		throw Rs232Exception("unable to set attributes");
	
	opened=true;
}
void Rs232::Close()
{
	close(fd);
	opened=false;
}

bool Rs232::Write(const unsigned char *buf, const int len)
{
	if(!opened)
		throw("write to closed pipe");
	return !(write(fd, buf, len)<0);
}

int Rs232::Read(unsigned char *buf, const int bufsize)
{
	if(!opened)
		throw("read from closed pipe");
	return read(fd, buf, bufsize);
}

int Rs232::ReadBlocking(unsigned char *buf, const int bufsize)
{
	if(!opened)
		throw("read from closed pipe");
	if(fcntl(fd,F_SETFL, 0)==-1)
		throw Rs232Exception("unable to set file flags");
	int len=read(fd, buf, bufsize);
	if(fcntl(fd,F_SETFL, FNDELAY)==-1)
		throw Rs232Exception("unable to set file flags");
	return len;
}

Rs232Exception::Rs232Exception(string newmsg)
{
	msg=newmsg;
}
string Rs232Exception::what()
{
	return msg;
}


} // end namespace
