/**
\file
\brief This file contains all code for compilation under WINDOWS. If you are developing for linux only, this file can be deleted.
\warning Do not compile this file directly if you design a cross-platform application. Compile rs232lib.cpp instead.
\author hephaisto

This file is licensed under the Creative Commons License (Attribution-NonCommercial) Version 3.0
http://creativecommons.org/licenses/by-nc/3.0/
*/
#include <sstream>

#include "rs232lib.h"

namespace rs232lib
{
using std::string;

Rs232::Rs232()
{
	opened=false;
}

void Rs232::Open(const string path, const int baudrate)
{
	if(opened)
		throw Rs232Exception("this instance of Rs232 has already an opened port");
	fd = CreateFileA(path.c_str(),
                      GENERIC_READ|GENERIC_WRITE,
                      0,                          /* no share  */
                      NULL,                       /* no security */
                      OPEN_EXISTING,
                      0,                          /* no threads */
                      NULL);                      /* no templates */
	if(fd==INVALID_HANDLE_VALUE)
		throw("unable to open port");
	
	DCB port_settings;
	memset(&port_settings, 0, sizeof(port_settings));
	port_settings.DCBlength = sizeof(port_settings);
	
	std::stringstream arguments;
	arguments<<"baud="<<baudrate<<" data=8 parity=N stop=1";
	if(!BuildCommDCBA(arguments.str().c_str(), &port_settings))
	{
		CloseHandle(fd);
		throw Rs232Exception("unable to read attributes");
	}
	
	if(!SetCommState(fd, &port_settings))
	{
		CloseHandle(fd);
		throw Rs232Exception("unable to set file flags");
	}
	
	COMMTIMEOUTS Cptimeouts;
	// first three values disable waitint for further characers (read returns immediately)
	Cptimeouts.ReadIntervalTimeout         = MAXDWORD;
	Cptimeouts.ReadTotalTimeoutMultiplier  = 0;
	Cptimeouts.ReadTotalTimeoutConstant    = 0;
	Cptimeouts.WriteTotalTimeoutMultiplier = 0;
	Cptimeouts.WriteTotalTimeoutConstant   = 0;

	if(!SetCommTimeouts(fd, &Cptimeouts))
	{
		CloseHandle(fd);
		throw Rs232Exception("unable to set timeout-settings");
	}
	
	opened=true;
}
void Rs232::Close()
{
	CloseHandle(fd);
	opened=false;
}

bool Rs232::Write(const unsigned char *buf, const int len)
{
	if(!opened)
		throw("write to closed pipe");

	int n;
	if(WriteFile(fd, buf, len, (LPDWORD)((void *)&n), NULL))
		return true;

	return false;
}

int Rs232::Read(unsigned char *buf, const int bufsize)
{
	if(!opened)
		throw("read from closed pipe");

	int len;
/*	if(size>4096)
		size = 4096;*/

	/* added the void pointer cast, otherwise gcc will complain about */
	/* "warning: dereferencing type-punned pointer will break strict aliasing rules" */
	ReadFile(fd, buf, bufsize, (LPDWORD)((void *)&len), NULL);
	return len;
}

int Rs232::ReadBlocking(unsigned char *buf, const int bufsize)
{
	if(!opened)
		throw("read from closed pipe");
	
	COMMTIMEOUTS Cptimeouts;
	Cptimeouts.ReadIntervalTimeout         = 0;
	Cptimeouts.ReadTotalTimeoutMultiplier  = 0;
	Cptimeouts.ReadTotalTimeoutConstant    = RS232LIB_READBLOCKING_MS;	// wait **ms for characters
	Cptimeouts.WriteTotalTimeoutMultiplier = 0;
	Cptimeouts.WriteTotalTimeoutConstant   = 0;

	if(!SetCommTimeouts(fd, &Cptimeouts))
		throw Rs232Exception("unable to set timeout-settings");
	
	int len;
	ReadFile(fd, buf, bufsize, (LPDWORD)((void *)&len), NULL);
	
	// first three values disable waitint for further characers (read returns immediately)
	Cptimeouts.ReadIntervalTimeout         = MAXDWORD;
	Cptimeouts.ReadTotalTimeoutMultiplier  = 0;
	Cptimeouts.ReadTotalTimeoutConstant    = 0;
	Cptimeouts.WriteTotalTimeoutMultiplier = 0;
	Cptimeouts.WriteTotalTimeoutConstant   = 0;

	if(!SetCommTimeouts(fd, &Cptimeouts))
	{
		opened=false;
		CloseHandle(fd);
		throw Rs232Exception("unable to set timeout-settings");
	}
	
	return len;
}

Rs232Exception::Rs232Exception(string newmsg)
{
	msg=newmsg;
}
string Rs232Exception::what()
{
	return msg;
}


} // end namespace
