/**
\file
\brief This file has to be included in all files that use rs232lib
\author hephaisto

This file is licensed under the Creative Commons License (Attribution-NonCommercial) Version 3.0
http://creativecommons.org/licenses/by-nc/3.0/

\todo include support for parity check, 2 stop-bits and non-8-bit-characters (currently only 8N1 is supported)
*/

#ifndef RS232LIB_INCLUDED
#define RS232LIB_INCLUDED


#include <string>

#ifndef __linux__
#include <windows.h>
#endif	// !__linux__


/// defines how long (time in ms) to wait for incoming data if fifo is empty (windows only!)
#define RS232LIB_READBLOCKING_MS 100


using std::string;

/// This namespace contains all classes supplied by this package
namespace rs232lib
{
//enum {RS232LIB_STOP_1=0, RS232LIB_STOP_15=1, RS232LIB_STOP_2=2, RS232LIB_PARITY_NO=0, RS232LIB_PARITY_ODD=4, RS232LIB_PARITY_EVEN=

/// Exception class for all actions in rs232lib
class Rs232Exception
{
private:
	string msg;
public:
	/// Constructor that takes the error message
	Rs232Exception(string newmsg);
	/// get the error message
	string what();
};

/// Class to handle all input and output via the serial port
class Rs232
{
private:
#ifdef __linux__
	int fd;
#else
	HANDLE fd;
#endif
	bool opened;
public:
	Rs232();
	/// Opens the specified serial port
	/**
	\param path "COM*" on windows, "/dev/tty*" on linux
	\param baudrate baudrate to use (linux currently only supports 1M)
	If opening failed for some reason, an Rs232Exception is raised
	\warning remember to close the port before you reopen it for another path!
	*/
	void Open(const string path, const int baudrate);
	/// Closes the serial port
	void Close();
	/// Writes a buffer to the port
	/**
	\param buf buffer to be sent
	\param len character count to send
	\return "true" if writing worked, "false" if writing failed
	
	If there is a fatal error (e.g. writing to a closed port), an Rs232Exception is raised
	*/
	bool Write(const unsigned char *buf, const int len);
	/// Reads from the port to the specified buffer
	/**
	\param buf Buffer to write data to
	\param bufsize maximum buffer size. Remember this library does not use any line-end characters ('\0').
	\return number of bytes read from the port; -1 if a serial error occured (linux interprets an empty fifo as error)
	
	If no data is present at the fifo, Read() immediately returns. If you want to wait for data, use ReadBlocking() instead.
	*/
	int Read(unsigned char *buf, const int bufsize);
	/// Reads from the port to the specified buffer
	/**
	Like Read(), except it waits for data to arrive.
	\warning under windows, ReadBlocking() waits only 100ms for data \see RS232LIB_READBLOCKING_MS
	*/
	int ReadBlocking(unsigned char *buf, const int bufsize);
};

}


/**
\example example.cpp

\section usage Usage
\subsection prep Preparation
- Include rs232lib.h to all files that use the library
- Add rs232lib.cpp to your list of source files (in your makefile or manually)

\subsection opening Using the port
- Open the port with Rs232::Open()
- Use Rs232::Read()/Write() to transmit data
- Close the port with Rs232::Close()

*/

/**
\mainpage
- This is a wrapper library for serial ports (RS232) on linux AND windows.
- The library is object-oriented and uses exceptions for nearly all error-conditions.
- The whole library is is licensed under the Creative Commons License (Attribution-NonCommercial) Version 3.0
http://creativecommons.org/licenses/by-nc/3.0/
- See the Examples page for a fast introduction
*/

#endif	// include-guards
